#include "general.h"
#include <unordered_map>
#include "Dialog/w3dexportdlg.h"
#include "EnumUtilities.h"
#include "resource.h"

extern HINSTANCE hInstance;

namespace
{
	W3D::MaxTools::W3DExportType RadioButtonIDToW3DExportType(uint16 controlID)
	{
		using namespace W3D::MaxTools;

		static const std::unordered_map<uint16, W3DExportType> s_dlg_id_map 
		{
			{ IDC_HIERARCHICAL_ANIMATED_MODEL,          W3DExportType::HierarchicalAnimatedModel},
			{ IDC_HIERARCHICAL_MODEL, W3DExportType::HierarchicalModel},
			{ IDC_PURE_ANIMATION,              W3DExportType::PureAnimation},
			{ IDC_SKELETON,                    W3DExportType::Skeleton},
			{ IDC_TERRAIN,                     W3DExportType::Terrain},
			{ IDC_SIMPLE_MESH,                 W3DExportType::SimpleMesh}
		};

		std::unordered_map<uint16, W3D::MaxTools::W3DExportType>::const_iterator search_it = s_dlg_id_map.find(controlID);

		return search_it == s_dlg_id_map.end() ? W3DExportType::Num : search_it->second;
	}

	uint16 W3DExportTypeToDialogID(W3D::MaxTools::W3DExportType exportType)
	{
		using namespace W3D::MaxTools;

		static const std::array<uint16, enum_to_value(W3DExportType::Num)> s_dlg_ids
		{
			IDD_W3D_EXPORT_HIERARCHICAL_MODEL,
			IDD_W3D_EXPORT_HIERARCHICAL_ANIMATED_MODEL,
			IDD_W3D_EXPORT_ANIMATION,
			IDD_W3D_EXPORT_SKELETON,
			IDD_W3D_EXPORT_TERRAIN,
			IDD_W3D_EXPORT_SIMPLE_MESH
		};

		return s_dlg_ids[enum_to_value(exportType)];
	}


	uint16 W3DExportTypeToRadioButtonID(W3D::MaxTools::W3DExportType exportType)
	{
		using namespace W3D::MaxTools;

		static const std::array<uint16, enum_to_value(W3DExportType::Num)> s_ctrl_ids
		{
			IDC_HIERARCHICAL_MODEL,
			IDC_HIERARCHICAL_ANIMATED_MODEL,
			IDC_PURE_ANIMATION,
			IDC_SKELETON,
			IDC_TERRAIN,
			IDC_SIMPLE_MESH
		};

		return s_ctrl_ids[enum_to_value(exportType)];
	}
}

namespace W3D::MaxTools
{
	W3DExportDlg::W3DExportDlg(W3DExportSettings & settings)
		: m_Settings(settings)
		, m_DialogRoot()
		, m_SettingsPanel()
		, m_ActiveTab()
		, m_Tabs()
	{ }

	INT_PTR W3DExportDlg::ShowDialog()
	{
		return DialogBoxParam(hInstance, MAKEINTRESOURCE(IDD_W3D_EXPORT), GetCOREInterface()->GetMAXHWnd(), W3DExportDlg::DlgProc, reinterpret_cast<LPARAM>(this));
	}

	INT_PTR W3DExportDlg::DlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
		W3DExportDlg* dlg = reinterpret_cast<W3DExportDlg*>(GetWindowLongPtr(hWnd, GWLP_USERDATA));

		switch (message)
		{
		case WM_INITDIALOG:
			dlg = reinterpret_cast<W3DExportDlg*>(lParam);
			SetWindowLongPtr(hWnd, GWLP_USERDATA, lParam);
			dlg->ConnectControls(hWnd);
			return TRUE;
		case WM_DESTROY:
			return TRUE;
		case WM_COMMAND:
			return dlg->HandleCommand(LOWORD(wParam), HIWORD(wParam));
		}
		return FALSE;
	}

	INT_PTR W3DExportDlg::TabProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
		W3DExportDlg* dlg = reinterpret_cast<W3DExportDlg*>(GetWindowLongPtr(hWnd, GWLP_USERDATA));

		switch (message)
		{
		case WM_INITDIALOG:
			dlg = reinterpret_cast<W3DExportDlg*>(lParam);
			SetWindowLongPtr(hWnd, GWLP_USERDATA, lParam);
			dlg->PopulateControls(hWnd);
			return TRUE;
		case WM_COMMAND:
			return dlg->HandleCommand(LOWORD(wParam), HIWORD(wParam));
		case CC_SPINNER_CHANGE:
			return dlg->HandleSpinner(LOWORD(wParam));
		}
		return FALSE;
	}

	void W3DExportDlg::ConnectControls(HWND root)
	{
		m_DialogRoot = root;
		m_SettingsPanel = GetDlgItem(root, IDC_SETTINGS_PANEL);
		
		RECT panelRect;
		GetClientRect(m_SettingsPanel, &panelRect);

		const uint32 dlu = GetDialogBaseUnits();
		const uint16 dlwidth = LOWORD(dlu);
		const uint16 dlheight = HIWORD(dlu);

		using BASE_TYPE = std::underlying_type_t<W3DExportType>;
		for (BASE_TYPE i = 0; i < m_Tabs.size(); ++i)
		{
			m_Tabs[i] = CreateDialogParam(hInstance, MAKEINTRESOURCE(W3DExportTypeToDialogID(static_cast<W3DExportType>(i))), m_SettingsPanel, TabProc, reinterpret_cast<LPARAM>(this));
			MoveWindow(m_Tabs[i], panelRect.left + dlwidth, panelRect.top + dlheight, (panelRect.right - panelRect.left) - dlwidth * 2, (panelRect.bottom - panelRect.top) - dlheight * 2, FALSE);
		}
		
		CheckRadioButton(m_DialogRoot, IDC_HIERARCHICAL_MODEL, IDC_SIMPLE_MESH, W3DExportTypeToRadioButtonID(m_Settings.ExportType));

		SetTab(m_Settings.ExportType);
	}

	void W3DExportDlg::PopulateControls(HWND root)
	{
		Interval animRange = GetCOREInterface()->GetAnimRange();
		int ticksPerFrame = GetTicksPerFrame();
		if (GetDlgItem(root, IDC_FRAMES_SPIN))
		{
			ReleaseISpinner(SetupIntSpinner(root, IDC_FRAMES_SPIN, IDC_FRAMES_EDIT, animRange.Start() / ticksPerFrame, animRange.End() / ticksPerFrame, m_Settings.AnimFramesStart));
		}
		if (GetDlgItem(root, IDC_FRAMES_TO_SPIN))
		{
			ReleaseISpinner(SetupIntSpinner(root, IDC_FRAMES_TO_SPIN, IDC_FRAMES_TO_EDIT, animRange.Start() / ticksPerFrame, animRange.End() / ticksPerFrame, m_Settings.AnimFramesEnd));
		}

		SetCheckBox(root, IDC_SMOOTH_VERTICES, m_Settings.SmoothVertexNormals);
		SetCheckBox(root, IDC_OPT_COLLISONS, m_Settings.OptimiseCollisions);
		SetCheckBox(root, IDC_USE_EXT_SKELETON, m_Settings.UseExistingSkeleton);
		SetCheckBox(m_DialogRoot, IDC_REVIEW_LOG, m_Settings.ReviewLog);
		
		RefreshExternalSkeletonButton();
	}

	void W3DExportDlg::SetTab(W3DExportType tab)
	{
		using BASE_TYPE = std::underlying_type_t<W3DExportType>;
		for (BASE_TYPE i = 0; i < m_Tabs.size(); ++i)
		{
			ShowWindow(m_Tabs[i], i == enum_to_value(tab) ? SW_SHOW : SW_HIDE);
		}

		m_ActiveTab = m_Tabs[enum_to_value(tab)];
		PopulateControls(m_ActiveTab);
	}

	void W3DExportDlg::SelectExternalSkeleton(HWND buttonHandle)
	{
		ICustButton* button = GetICustButton(buttonHandle);
		static FilterList s_fl = []() { FilterList fl; fl.Append(_M("W3D File (*.w3d)")); fl.Append(_M("*.w3d")); return
			fl; }();
		MSTR fname(m_Settings.ExistingSkeletonFileName);
		MSTR dir(m_Settings.ExistingSkeletonFileDirectory);
		if (GetCOREInterface16()->DoMaxOpenDialog(GetCOREInterface()->GetMAXHWnd(), _M("Existing Skeleton"), fname, dir, s_fl))
		{
			wcscpy(m_Settings.ExistingSkeletonFileName, fname);
			wcscpy(m_Settings.ExistingSkeletonFileDirectory, dir);
			RefreshExternalSkeletonButton();
		}

		ReleaseICustButton(button);
	}

	void W3DExportDlg::RefreshExternalSkeletonButton()
	{
		HWND skelBrowse = GetDlgItem(m_ActiveTab, IDC_BROWSE);
		if (skelBrowse)
		{
			ICustButton* btn = GetICustButton(skelBrowse);

			if (m_Settings.ExistingSkeletonFileName[0])
			{
				MSTR filename;
				SplitFilename(m_Settings.ExistingSkeletonFileName, nullptr, &filename, nullptr);
				btn->SetText(filename);
			}
			else
			{
				btn->SetText(_M("Browse..."));
			}
			btn->Enable(m_Settings.UseExistingSkeleton);
			ReleaseICustButton(btn);
		}
	}

	INT_PTR W3DExportDlg::HandleCommand(uint16 controlID, uint16 commandID)
	{
		switch (controlID)
		{
		case IDC_SMOOTH_VERTICES:
			m_Settings.SmoothVertexNormals = IsDlgButtonChecked(m_ActiveTab, controlID);
			return TRUE;
		case IDC_OPT_COLLISONS:
			m_Settings.OptimiseCollisions = IsDlgButtonChecked(m_ActiveTab, controlID);
			return TRUE;
		case IDC_USE_EXT_SKELETON:
			m_Settings.UseExistingSkeleton = IsDlgButtonChecked(m_ActiveTab, controlID);
			RefreshExternalSkeletonButton();
			return TRUE;
		case IDC_BROWSE:
			SelectExternalSkeleton(GetDlgItem(m_ActiveTab, controlID));
			return TRUE;
		case IDC_REVIEW_LOG:
			m_Settings.ReviewLog = IsDlgButtonChecked(m_DialogRoot, controlID);
			return TRUE;
		
		//Tab Selection
		case IDC_HIERARCHICAL_MODEL:
		case IDC_HIERARCHICAL_ANIMATED_MODEL:
		case IDC_PURE_ANIMATION:
		case IDC_SKELETON:
		case IDC_TERRAIN:
		case IDC_SIMPLE_MESH:
			m_Settings.ExportType = RadioButtonIDToW3DExportType(controlID);
			SetTab(m_Settings.ExportType);
			return TRUE;

		//Close Commands
		case IDOK:
		case IDCANCEL:
		case IDCLOSE:
			EndDialog(m_DialogRoot, controlID);
			return TRUE;
		}

		return FALSE;
	}

	INT_PTR W3DExportDlg::HandleSpinner(uint16 controlID)
	{
		switch (controlID)
		{
		case IDC_FRAMES_SPIN:
		{
			ISpinnerControl* sc = GetISpinner(GetDlgItem(m_ActiveTab, controlID));
			m_Settings.AnimFramesStart = sc->GetIVal();
			ReleaseISpinner(sc);
			return TRUE;
		}
		case IDC_FRAMES_TO_SPIN:
		{
			ISpinnerControl* sc = GetISpinner(GetDlgItem(m_ActiveTab, controlID));
			m_Settings.AnimFramesEnd = sc->GetIVal();
			ReleaseISpinner(sc);
			return TRUE;
		}
		}
		return FALSE;
	}
}